//******************************************************************************
//  MSP430F20xx Demo - I2C Master Transmitter and Receiver for bq27500 EVM
//
//  Description: MSP430 I2C Master communicates with a bq27500EVM (I2C Slave)
//  using an I2C Master software driver.  Whenever master device transmits data
//  to the slave device, the data is read back for verification purposes.
//  If the data is incorrect, the program will toggle the P1.0 LED forever.
//  The various LED's on the bq27500EVM are enabled/disabled to demonstrate
//  I2C communications.
//
//  ACLK = n/a, MCLK = SMCLK = Calibrated 1MHz
//
//                   Slave                      Master
//                  bq27500                  MSP430F20xx
//             -----------------   /|\    -----------------
//            |                 |   |    |              XIN|-
//            |                 |   |    |                 |
//            |                 |   |    |             XOUT|-
//            |                 |  10K   |                 |
//            |                 |   |    |                 |
//            |                 |   |    |             P1.0|---> LED
//            |              SDA|---|--->|SDA              |
//            |              SCL|<--|----|SCL              |
//
//  Note: Internal pull-ups may be used in this example for SDA & SCL
//        bq27500 7-bit Slave address = 1010101b
//
//  R. Wu
//  Texas Instruments Inc.
//  January 2010
//  Built with IAR Embedded Workbench Version: 4.21A
//******************************************************************************

#include <msp430x20x2.h>
#include "bq27500.h"                                // Device-specific header
#include "..\..\lib\I2C_USI\USI_I2CMaster.h"        // For USI module
#include "..\..\lib\I2C_GPIO\MSP430_SWI2C_Master.h" // For GPIO bit-banging

/**** USER CONFIGURATION: Select ONE of the following I2C physical layers *****/
#define _I2C_USI                            // OPTION #1: Use MSP430 USI module
//#define _I2C_BITBANG                        // OPTION #2: Use Standard GPIO pins

#define ATRATE_MA            -100           // USER CONFIG: AtRate setting (mA)
#define I2CSLAVEADDR         0x55           // 7-bit slave address
#define BUFFERSIZE             32           // # of bytes for Tx & Rx buffers

unsigned char TxData[BUFFERSIZE];           // Stores data bytes to be TX'd
unsigned char RxData[BUFFERSIZE];           // Stores data bytes that are RX'd
unsigned int  temperature;                  // Stores temperature
unsigned int  voltage;                      // Stores voltage
  signed int  atrate;                       // Stores AtRate
unsigned int  artte;                        // Stores AtRate Time to Empty
unsigned int  soc;                          // Stores State of Charge
unsigned int  dcap;                         // Stores Design Capacity
unsigned int  dnamelen;                     // Stores Device Name Length

int StatusCallback(unsigned char c);
unsigned int transBytes2Int(unsigned char msb, unsigned char lsb);
void MSP430_bq27500_read(unsigned char cmd, unsigned int bytes);
void MSP430_bq27500_write(unsigned char cmd, unsigned char data);
void MSP430_bq27500_error(void);

unsigned int transBytes2Int(unsigned char msb, unsigned char lsb)
{
  unsigned int tmp;
  
  tmp = ((msb << 8) & 0xFF00);
  return ((unsigned int)(tmp + lsb) & 0x0000FFFF);  
}

#ifdef _I2C_USI
int StatusCallback(unsigned char c)
{
 return TI_USI_EXIT_LPM;                    // Exit active for next transfer
}
#endif /* _I2C_USI */

void MSP430_bq27500_read(unsigned char cmd, unsigned int bytes)
{
  unsigned char tx[1];
  
  tx[0] = cmd;
#ifdef _I2C_USI
  __disable_interrupt();
  TI_USI_I2CWrite(I2CSLAVEADDR, 1, 1, tx);
  __bis_SR_register(LPM0_bits + GIE);
  __disable_interrupt();
  TI_USI_I2CRead(I2CSLAVEADDR, bytes, 0, RxData); 
  __bis_SR_register(LPM0_bits + GIE);
#endif /* _I2C_USI */
#ifdef _I2C_BITBANG
  MSP430_SWI2CMST_writeBlock(I2CSLAVEADDR, 1, 1, tx);
  MSP430_SWI2CMST_readBlock(I2CSLAVEADDR, bytes, RxData);
#endif /* _I2C_BITBANG */
}

void MSP430_bq27500_write(unsigned char cmd, unsigned char data)
{
  unsigned char tx[2];
  
  tx[0] = cmd;
  tx[1] = data;
#ifdef _I2C_USI
  __disable_interrupt();
  TI_USI_I2CWrite(I2CSLAVEADDR, 2, 0, tx);
  __bis_SR_register(LPM0_bits + GIE);
#endif /* _I2C_USI */
#ifdef _I2C_BITBANG
  MSP430_SWI2CMST_writeBlock(I2CSLAVEADDR, 2, 0, tx);
#endif /* _I2C_BITBANG */
  __delay_cycles(50000);
}

void MSP430_bq27500_error(void)
{
  while (1)                                 // Loop forever
  {
    P1OUT ^= BIT0;                          // Toggle LED
    __delay_cycles(50000);                  // Delay
  }
}

void main(void)
{ 
  unsigned int i;
  unsigned int sum = 0;
  unsigned char checksum;
  unsigned char msb, lsb;
  
  WDTCTL = (WDTPW | WDTHOLD);               // Disable Watchdog
  BCSCTL1 = CALBC1_1MHZ;                    // Set DCO to 1 MHz
  DCOCTL = CALDCO_1MHZ;
  P1DIR |= BIT0;                            // P1.0 output direction
  P1OUT &= ~BIT0;                           // Turn P1.0 LED OFF
#ifdef _I2C_BITBANG
  MSP430_SWI2CMST_init();                   // Initialize I2C pins
#endif /* _I2C_BITBANG */
  
#ifdef _I2C_USI
  // Initialize USI module, clock = ~(SMCLK/32) 
  TI_USI_I2C_MasterInit((USIDIV_5 | USISSEL_2 | USICKPL), StatusCallback);
  
  // Acknowledge polling function - LED blinks continuously until slave device 
  // provides an ACK
  while (TI_USI_I2CSelect(I2CSLAVEADDR))
  { 
    P1OUT ^= BIT0;                          // Toggle LED
    __delay_cycles(50000);                  // Delay    
  }
  P1OUT = 0x00;                             // Slave acknowledged, LED off
#endif /* _I2C_USI */
  
  // Read Temperature (units = 0.1K)
  MSP430_bq27500_read(bq27500CMD_TEMP_LSB, 2);
  temperature = transBytes2Int(RxData[1], RxData[0]);
    
  // Read Voltage (units = mV)
  MSP430_bq27500_read(bq27500CMD_VOLT_LSB, 2);
  voltage = transBytes2Int(RxData[1], RxData[0]);
  
  // Set AtRate (units = mA)
  atrate = ATRATE_MA;
  msb = ((atrate >> 8) & 0x00FF);
  lsb = (atrate & 0x00FF);
  MSP430_bq27500_write(bq27500CMD_AR_LSB, lsb); // Must write the LSB before MSB
  MSP430_bq27500_write(bq27500CMD_AR_MSB, msb);

  // Verify AtRate was set correctly (units = mA)
  MSP430_bq27500_read(bq27500CMD_AR_LSB, 2);
  if ((RxData[1] != msb) || (RxData[0] != lsb))
  {
    MSP430_bq27500_error();                 // Signal error condition occurred
  }
  
  // Read AtRate Time to Empty (units = Minutes)
  MSP430_bq27500_read(bq27500CMD_ARTTE_LSB, 2);
  artte = transBytes2Int(RxData[1], RxData[0]);
  
  // Read State of Charge (units = %)
  MSP430_bq27500_read(bq27500CMD_SOC_LSB, 2);
  soc = transBytes2Int(RxData[1], RxData[0]);

  // Read Design Capacity (units = mAH)
  MSP430_bq27500_read(bq27500CMD_DCAP_LSB, 2);
  dcap = transBytes2Int(RxData[1], RxData[0]);
  
  // Read Device Name Length
  MSP430_bq27500_read(bq27500CMD_DNAMELEN, 1);
  dnamelen = RxData[0];
  
  // Read Device Name (Rx buffer should end up with ASCII chars for 'bq27500')
  MSP430_bq27500_read(bq27500CMD_DNAME, 7);
  
  // Write & read back 32 bytes of data in Manufacturer Info Block A
  for (i = 0; i < BUFFERSIZE; i++)          
  {
    TxData[i] = i;                          // Initialize data to be written
  }
  MSP430_bq27500_write(bq27500CMD_DFDCNTL, 0);// BlockDataControl() = 0x00
  MSP430_bq27500_write(bq27500CMD_DFCLS, 58);// Write the subclass value
  MSP430_bq27500_write(bq27500CMD_DFBLK, 0);// Select offset within the flash  
  for (i = 0; i < BUFFERSIZE; i++)          // Compute the checksum of the block
  {
    sum += TxData[i];                       // Calculate the sum of the values  
  }
  checksum = (0xFF - (sum & 0x00FF));       // Compute checksum based on the sum
  MSP430_bq27500_write(bq27500CMD_DFDCKS, checksum);// Write the checksum value
  for (i = 0; i < BUFFERSIZE; i++)          // Write 32 bytes to Info Block A
  {
    MSP430_bq27500_write((bq27500CMD_ADF+i), TxData[i]);  
  }  
  
  MSP430_bq27500_read(bq27500CMD_ADF, BUFFERSIZE);// Read the contents of block
  for (i = 0; i < BUFFERSIZE; i++)          // Check if writes were successful
  {
    if (TxData[i] != RxData[i])             // Tx & Rx data values match?
    {
      MSP430_bq27500_error();               // Signal error condition occurred
    }
  }
  
  // Set P1.0 LED on MSP430 EVM to signal that command sequence was successful
  P1OUT |= BIT0; 
}
